/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#include <glib.h>
#include <stdlib.h>
#include <string.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "nb_noteList.h"

notelist *
nb_notelist_new	(const gchar * filename)
{
	notelist * nl = g_malloc (sizeof (notelist));
	nl->notes = g_ptr_array_new ();

	nl->filename = (gchar *) filename;

	return (nl);
}

void
nb_notelist_add	(notelist * nl, note * n)
{
	g_ptr_array_add (nl->notes, n);
}

void
nb_notelist_remove (notelist * nl, note * n)
{
	g_ptr_array_remove (nl->notes, n);
}

void 
nb_notelist_free (notelist * nl)
{
	g_ptr_array_free (nl->notes, 1);

	g_free (nl->filename); 
}

gboolean
nb_notelist_load_from_file (notelist * nl)
{
	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseFile (nl->filename);

	if ( doc == NULL )
	{
		fprintf (stderr, "Document not parsed successfully\n"); 
		return FALSE;
	}

	cur = xmlDocGetRootElement (doc);

	if ( cur == NULL )
	{
		fprintf (stderr, "empty document\n");
		xmlFreeDoc (doc);
		return FALSE;
	}

	if ( xmlStrcmp (cur->name, (const xmlChar *) "notelist") )
	{
		fprintf (stderr, "wrong document type, root node is not \"notelist\"\n");
		xmlFreeDoc (doc);
		return FALSE;
	}

	cur = cur->xmlChildrenNode;

	/* loop every "note" node */
	while ( cur != NULL )
	{
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "note") )
		{
			note * n = NULL;
			n = nb_note_load_from_file (doc, cur);

			if ( n == NULL )
			{
				fprintf (stderr, "invalid note in file, skip note...\n");
			}
			else
			{
				/* add the note to the current notelist */
				nb_notelist_add (nl, n);
			}

		}

		cur = cur->next;
	}

	xmlFreeDoc (doc);	

	return TRUE;
}

int
nb_notelist_get_note_index (notelist * nl, note * n)
{
	int i;

	for ( i = 0 ; i < nl->notes->len ; i++ )
	{
		if ( g_ptr_array_index (nl->notes, i) == n )
			return i;
	}

	return -1;
}

gchar *
nb_notelist_get_text_from_index (notelist * nl, int index)
{
	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseFile (nl->filename);

	if ( doc == NULL )
	{
		fprintf (stderr, "Document not parsed successfully\n"); 
		return NULL;
	}

	cur = xmlDocGetRootElement (doc);

	if ( cur == NULL )
	{
		fprintf (stderr, "empty document\n");
		xmlFreeDoc (doc);
		return NULL;
	}

	if ( xmlStrcmp (cur->name, (const xmlChar *) "notelist") )
	{
		fprintf (stderr, "wrong document type, root node is not \"notelist\"\n");
		xmlFreeDoc (doc);
		return NULL;
	}

	cur = cur->xmlChildrenNode;

	/* loop every "note" node */
	while ( cur != NULL )
	{
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "note") )
		{
			int i;

			i = atoi (xmlGetProp (cur, "index"));

			if ( i == index )
			{
				xmlChar * text;
				text = nb_note_get_text_from_file (doc, cur);

				xmlFreeDoc (doc);

				return text;
			}
		}

		cur = cur->next;
	}

	xmlFreeDoc (doc);	

	return NULL;
}

gboolean
nb_notelist_update_file (notelist * nl, note * n, int action)
{
	xmlDocPtr doc;
	xmlNodePtr cur;
	xmlNsPtr ns;
	int i = 0;


	doc = xmlParseFile (nl->filename);

	if ( doc == NULL )
	{
		if ( action == ADD_NOTE )
		{
			printf ("No document found, creating one.\n");

			/* document is empty */
			doc = xmlNewDoc ("1.0");

			cur = xmlNewNode (NULL, "notelist");


			xmlDocSetRootElement (doc, cur);
		}
		else
		{
			fprintf (stderr, "Document not parsed successfully\n"); 
			return FALSE;
		}
	}

	cur = xmlDocGetRootElement (doc);

	ns = xmlNewNs (cur, "http://www.notabene.org/ns", "notabene");

	if ( cur == NULL )
	{
		fprintf (stderr, "empty document\n");
		xmlFreeDoc (doc);
		return FALSE;
	}

	if ( xmlStrcmp (cur->name, (const xmlChar *) "notelist") )
	{
		fprintf (stderr, "wrong document type, root node is not \"notelist\"\n");
		xmlFreeDoc (doc);
		return FALSE;
	}

	if ( action == ADD_NOTE )
	{
		xmlAddChild (cur, nb_note_get_new_pointer (n, ns, nl->notes->len));
	}
	else
	{
		gchar increase = 0;
		int index;

		cur = cur->xmlChildrenNode;

		index = nb_notelist_get_note_index (nl, n);

		/* loop every "note" node */
		while ( cur != NULL )
		{
			if ( !xmlStrcmp (cur->name, (const xmlChar *) "note") )
			{

				i = atoi (xmlGetProp (cur, "index"));

				if ( increase )
				{
					gchar cindex[5];

					snprintf (cindex, 4, "%d", i - 1);
					xmlSetProp (cur, "index", cindex);
				}
				else if ( i == index )
				{	

					if ( action == UPDATE_NOTE )
					{
						xmlReplaceNode (cur, nb_note_get_new_pointer (n, ns, index));
					}
					else if ( action == DEL_NOTE )
					{
						xmlNodePtr del;

						del = cur;

						if ( cur->prev != NULL )
							cur->prev->next = cur->next;
						if ( cur->next != NULL )
							cur->next->prev = cur->prev;

						xmlFreeNode (del);
						increase = 1;
					}
				}
			}

			cur = cur->next;
		}
	}

	xmlSaveFormatFile (nl->filename, doc, 1);
	xmlFreeDoc (doc);

	return TRUE;

}
